// This file is to support: Wayland retrace

#include <stdio.h>
#include <stdlib.h>
#include <iostream>
#include <fstream>
#include <dlfcn.h>
#include "glproc.hpp"
#include "glws.hpp"
#include <unistd.h>

#include "string.h"
#include <wayland-client.h>
#include <wayland-egl.h>
#include <compositor-shim.h>
#include "wayland-util.h"	

#ifndef ARRAY_LENGTH
#define ARRAY_LENGTH(a) (sizeof (a) / sizeof (a)[0])
#endif

//#define DEBUG_ENABLE

#ifdef DEBUG_ENABLE
#define DEBUG_ENTER() std::cerr << "Entering " << __FUNCTION__ << "\n"
#define DEBUG_LEAVE() std::cerr << "Leaving " << __FUNCTION__ << "\n"
#else
#define DEBUG_ENTER() /*No-op*/
#define DEBUG_LEAVE() /*No-op*/
#endif

typedef struct _WaylandGles WaylandGles;

extern unsigned int layer_id;
extern unsigned int surface_id;
extern unsigned int surf_width;
extern unsigned int surf_height;
extern unsigned char pixmap_to_display;

unsigned int use_wayland_backend;

extern "C"{
void* fbCreatePixmap(void* Display, int Width, int Height);
void fbDestroyPixmap(void* Pixmap);
}

struct _WaylandGles
{
  unsigned int surface_id;
  unsigned int layer_id;
  struct wl_display* wl_display;
  struct wl_surface* wl_surface;
  struct wl_egl_window* window;
  struct wl_compositor* wl_compositor;
  struct serverinfo* wl_ext_serverinfo;
  unsigned int wl_connect_id;
  struct wl_registry* wl_registry;
  EGLDisplay eglDisplay;
  EGLSurface eglSurface;

  struct compositor_shim_context* wlAdapterContext;
  struct compositor_shim_surface_context wlAdapterSurfaceContext;

};

namespace glws {
static WaylandGles _egl;
static char const *eglExtensions = NULL;
static bool has_EGL_KHR_create_context = false;

static const struct wl_interface *types[] = {
	NULL,
};

static const struct wl_message serverinfo_requests[] = {
	{ "get_connection_id", "", types + 0 },
};

static const struct wl_message serverinfo_events[] = {
	{ "connection_id", "u", types + 0 },
};

const struct wl_interface serverinfo_interface = {
	"serverinfo", 1,
	ARRAY_LENGTH(serverinfo_requests), serverinfo_requests,
	ARRAY_LENGTH(serverinfo_events), serverinfo_events,
};

static void serverinfo_cb_impl(void *data, struct serverinfo *pServerinfo, uint32_t client_handle);

static void RegistryHandleGlobal(void* data, struct wl_registry* registry, uint32_t name, const char* interface, uint32_t version);

static struct wl_registry_listener registryListener = {
    RegistryHandleGlobal,
    NULL
};

struct serverinfo_listener {
	void (*connection_id)(void *data,
			      struct serverinfo *serverinfo,
			      uint32_t connection_id);
};

struct serverinfo_listener serverinfo_cb = {
    serverinfo_cb_impl
};


static EGLenum
translateAPI(glfeatures::Profile profile)
{
    switch (profile.api) {
    case glfeatures::API_GL:
        return EGL_OPENGL_API;
    case glfeatures::API_GLES:
        return EGL_OPENGL_ES_API;
    default:
        assert(0);
        return EGL_NONE;
    }
}


/* Must be called before
 *
 * - eglCreateContext
 * - eglGetCurrentContext
 * - eglGetCurrentDisplay
 * - eglGetCurrentSurface
 * - eglMakeCurrent (when its ctx parameter is EGL_NO_CONTEXT ),
 * - eglWaitClient
 * - eglWaitNative
 */
static void
bindAPI(EGLenum api)
{
    if (eglBindAPI(api) != EGL_TRUE) {
        std::cerr << "error: eglBindAPI failed\n";
        exit(1);
    }
}


class WlVisual : public Visual
{
public:
    EGLConfig config;

    WlVisual(Profile prof, EGLConfig cfg) :
        Visual(prof),
        config(cfg)
    {}

    virtual ~WlVisual() {
    }
};

class WlDrawable : public Drawable
{
public:
    EGLenum api;
	EGLNativePixmapType native_pixmap;
	EGLSurface eglSurface;

    WlDrawable(const Visual *vis, int w, int h, 
	const glws::pbuffer_info *pbInfo) :
        Drawable (vis, w, h, pbInfo),
        api(EGL_OPENGL_ES_API)
    {
        create(w, h, true, pbuffer);
    }

    virtual ~WlDrawable()
    {
        destroy();
    }

    void
    create(int w, int h, bool createEglSurf = true, bool pbuffer = false) {		
		if(pbuffer && !pixmap_to_display)
		{				
		//	native_pixmap = fbCreatePixmap(_egl.eglDisplay, surf_width, surf_height);
			EGLConfig config = static_cast<const WlVisual *>(visual)->config;
			eglSurface = eglCreatePixmapSurface(_egl.eglDisplay, config, native_pixmap, NULL);
			if(eglSurface == EGL_NO_SURFACE) {
	           std::cerr << "error: failed to create pixmap surface:" << eglGetError() << "\n";				  				   
	           exit(1);
	        }
		}
		else
		{				
		    if(createEglSurf) {
	            eglWaitNative(EGL_CORE_NATIVE_ENGINE);
	            EGLConfig config = static_cast<const WlVisual *>(visual)->config;
                if(use_wayland_backend)
	            {
                    eglSurface = eglCreateWindowSurface(_egl.eglDisplay, config, (EGLNativeWindowType)_egl.window, NULL);
	            }
	            else
	            {
	                eglSurface = eglCreateWindowSurface(_egl.eglDisplay, config, (EGLNativeWindowType)0, NULL);
	            }
	            if(eglSurface == EGL_NO_SURFACE) {
	                std::cerr << "error: failed to create window surface\n";
	                exit(1);
	            }
	        }			
			native_pixmap = NULL;
		}
    }

    void
    destroy(bool destroyEglSurf = true) {   	
		if(destroyEglSurf) {
			if(native_pixmap){
			//	fbDestroyPixmap(native_pixmap);
				native_pixmap = NULL;
			}
			
#ifdef ENABLE_DEBUG				
			std::cerr << "Destroy eglSurface" << "\n";
#endif
	        if(eglSurface != EGL_NO_SURFACE) {
	            eglDestroySurface(_egl.eglDisplay, eglSurface);
	            eglWaitClient();
	            eglSurface = EGL_NO_SURFACE;
	        }
	     }
    }

    void
    resize(int w, int h) {
        if (w == width && h == height) {
            return;
        }		

        destroy(false);
        create(w, h, false);

        Drawable::resize(w, h);
    }

    void
    show(void) {
        if (visible) {
            return;
        }
        Drawable::show();
    }

    void
    swapBuffers(void) {
    	eglSwapBuffers(_egl.eglDisplay, eglSurface);
    }   
};

class WlContext : public Context
{
public:
    EGLContext context;

    WlContext(const Visual *vis, EGLContext ctx) :
        Context(vis),
        context(ctx)
    {}

    virtual ~WlContext() {
		eglDestroyContext(_egl.eglDisplay, context);	
    }
};

/**
 * Load the symbols from the specified shared object into global namespace, so
 * that they can be later found by dlsym(RTLD_NEXT, ...);
 */
static void
load(const char *filename)
{
    if (!dlopen(filename, RTLD_GLOBAL | RTLD_LAZY)) {
        std::cerr << "error: unable to open " << filename << "\n";
        exit(1);
    }
}

static void serverinfo_cb_impl(void *data, struct serverinfo *pServerinfo, uint32_t client_handle)
{
    pServerinfo = pServerinfo;

    WaylandGles * sink = (WaylandGles *)data;
    sink->wl_connect_id = client_handle;
}

#define SERVERINFO_GET_CONNECTION_ID	0
static void RegistryHandleGlobal(void* data, struct wl_registry* registry, uint32_t name, const char* interface, uint32_t version)
{
  version = version;
  WaylandGles *sink = (WaylandGles*)data;

	if (!strcmp(interface, "wl_compositor")){
	    sink->wl_compositor =   (struct wl_compositor*)(wl_registry_bind(registry,
										  name,
										  &wl_compositor_interface,
										  1));
	}

	if (!strcmp(interface, "serverinfo")){
	    sink->wl_ext_serverinfo = (struct serverinfo*)wl_registry_bind( registry, name, &serverinfo_interface, 1);

		wl_proxy_add_listener((struct wl_proxy *) sink->wl_ext_serverinfo,
						     (void (**)(void)) &serverinfo_cb, data);

		wl_proxy_marshal((struct wl_proxy *) sink->wl_ext_serverinfo,
						 SERVERINFO_GET_CONNECTION_ID);
	}
}

void
init(void) {
    DEBUG_ENTER();

    load("libEGL.so.1");

    //parse powervr.ini for WindowSystem
    {
        std::ifstream pvrfile("/etc/powervr.ini");
        std::string pvrline;

        if(!pvrfile) {
            printf("WINDOW SYSTEM: Could not find powervr.ini file. Window system set to Wayland backend.\n");
            use_wayland_backend = 1;
        } else {
            while(std::getline(pvrfile, pvrline)) {
                if (pvrline[0] != ';' && pvrline.find("WindowSystem") != std::string::npos) {
                    if(pvrline.find("libpvrWAYLAND_WSEGL.so") != std::string::npos) {
                        printf("WINDOW SYSTEM: Switched to Wayland\nWINDOW SYSTEM: From powervr.ini: %s\n", pvrline.c_str());
                        use_wayland_backend = 1;
                        break;
                    } else if (pvrline.find("libpvrDRM_WSEGL.so") != std::string::npos) {
                        printf("WINDOW SYSTEM: Switched to DRM\nWINDOW SYSTEM: From powervr.ini: %s\n", pvrline.c_str());
                        use_wayland_backend = 0;
                        break;
                    } else {
                        printf("WINDOW SYSTEM: Could not find settings for window system in powervr.ini file. Window system set to Wayland backend.\n");
                        use_wayland_backend = 1;
                        break;
                    }
                }
            }
        }
    }

	//wayland setup
	{
		/* set wayland environment to run without shell script */		
		if(NULL == getenv("XDG_RUNTIME_DIR")){
		  setenv("XDG_RUNTIME_DIR","/tmp/", 1);
		}
		_egl.surface_id = surface_id;
	  	_egl.layer_id = layer_id;

		/*wayland init*/
        if(use_wayland_backend)
        {
            _egl.wl_display = wl_display_connect(NULL);
			if (_egl.wl_display == NULL){
				std::cerr << "error: failed to create wl_display\n";
				exit(1);
			}


			_egl.wl_registry = wl_display_get_registry(_egl.wl_display);
			wl_registry_add_listener(_egl.wl_registry, &registryListener, &_egl);
			wl_display_dispatch(_egl.wl_display);
			wl_display_roundtrip(_egl.wl_display);
			if(_egl.wl_compositor == NULL){
				std::cerr << "error: failed to create wl_compositor\n";
				exit(1);
			}

			_egl.wlAdapterContext = compositor_shim_initialize (_egl.wl_display);
			if (NULL == _egl.wlAdapterContext)
			{
				std::cerr << "error: failed to initialize compositor shim\n";
				exit(1);
			}

			_egl.wl_surface = wl_compositor_create_surface(_egl.wl_compositor);
			if(_egl.wl_surface == NULL){
				std::cerr << "error: failed to create wl_surface\n";
				exit(1);
			}

			_egl.window = wl_egl_window_create(_egl.wl_surface, surf_width, surf_height);
			if (_egl.window == NULL){
				std::cerr << "error: failed to create wl_window\n";
				exit(1);
			}

			int adapter_error = 0;

			adapter_error = compositor_shim_surface_init (&_egl.wlAdapterSurfaceContext, _egl.wl_surface, _egl.layer_id,
													_egl.surface_id,surf_width, surf_height);
			if (adapter_error != 0)
			{
				printf("compositor_shim_surface_init() failed\n");
				return;
			}

			adapter_error = compositor_shim_surface_configure (_egl.wlAdapterContext, &_egl.wlAdapterSurfaceContext, ADAPTER_CONFIGURATION_ALL);
			if (adapter_error != 0)
			{
				printf("compositor_shim_surface_configure() failed\n");
				return;
			}
			_egl.eglDisplay = eglGetDisplay(_egl.wl_display);
        }
        else
        {
            _egl.eglDisplay = eglGetDisplay((void *)1);
        }

		if (_egl.eglDisplay == EGL_NO_DISPLAY) {
		      std::cerr << "error: unable to get EGL display\n";
		      exit(1);
		}

	}

    EGLint major, minor;
    if (!eglInitialize(_egl.eglDisplay, &major, &minor)) {
        std::cerr << "error: unable to initialize EGL display\n";
        exit(1);
    }

    eglExtensions = eglQueryString(_egl.eglDisplay, EGL_EXTENSIONS);
    has_EGL_KHR_create_context = checkExtension("EGL_KHR_create_context", eglExtensions);	

    DEBUG_LEAVE();
}

void
cleanup(void) {
    DEBUG_ENTER();
	
    eglWaitNative(EGL_CORE_NATIVE_ENGINE);

    if (_egl.eglDisplay != EGL_NO_DISPLAY) {
        eglTerminate(_egl.eglDisplay);
    }

	//wayland terminate
    if(use_wayland_backend)
	{
		wl_display_flush(_egl.wl_display);
	  	wl_display_roundtrip( _egl.wl_display);
		compositor_shim_terminate(_egl.wlAdapterContext);

		if(_egl.window != NULL)
             wl_egl_window_destroy(_egl.window);

		_egl.window = NULL;

		if (_egl.wl_surface)
		     wl_surface_destroy(_egl.wl_surface);

		if (_egl.wl_compositor)
		     wl_compositor_destroy(_egl.wl_compositor);



		if (_egl.wl_display)
			wl_display_disconnect(_egl.wl_display);
	}

    DEBUG_LEAVE();
}

Config*
chooseConfig(unsigned int *attrib_array) {
    // EGL configuration - we use 24-bpp render target and a 16-bit Z buffer.
    EGLint configAttribs[] =
    {
        EGL_SAMPLES,      0,
        EGL_RED_SIZE,     8,
        EGL_GREEN_SIZE,   8,
        EGL_BLUE_SIZE,    8,
        EGL_ALPHA_SIZE,   EGL_DONT_CARE,
        EGL_DEPTH_SIZE,   8,
        EGL_RENDERABLE_TYPE,    EGL_OPENGL_ES2_BIT,
#if GPU_HW_VENDOR_VIVANTE
        EGL_SURFACE_TYPE, EGL_WINDOW_BIT|EGL_PIXMAP_BIT,
#else
		EGL_SURFACE_TYPE, EGL_WINDOW_BIT,
#endif
        EGL_NONE,
    };

    EGLConfig config;
    EGLint matchingConfigs;
    Config *cnfg = NULL;
    EGLint num_config;

    eglGetConfigs(_egl.eglDisplay, NULL, 0, &num_config);

    std::cout << "num_configs:" << num_config << "\n";

    if(attrib_array && eglChooseConfig(_egl.eglDisplay, (EGLint*)attrib_array, &config, \
        1, &matchingConfigs) && (matchingConfigs >= 1))
    {
        std::cout << "chooseConfig: the application's config attributes used for creating config\n";
        cnfg = new Config((intptr_t)config);
    }
    else
    {
        eglChooseConfig(_egl.eglDisplay,
                        configAttribs,
                        &config,
                        1,
                        &matchingConfigs);
        if(matchingConfigs >= 1)
            cnfg = new Config((intptr_t)config);
    }
    return cnfg;
}

static EGLConfig
findEglConfig() {
    // EGL configuration - we use 24-bpp render target and a 16-bit Z buffer.
    EGLint configAttribs[] =
    {
        EGL_SAMPLES,      0,
        EGL_RED_SIZE,     8,
        EGL_GREEN_SIZE,   8,
        EGL_BLUE_SIZE,    8,
        EGL_ALPHA_SIZE,   EGL_DONT_CARE,
        EGL_DEPTH_SIZE,   8,
        EGL_RENDERABLE_TYPE,    EGL_OPENGL_ES2_BIT,
#if GPU_HW_VENDOR_VIVANTE
        EGL_SURFACE_TYPE, EGL_WINDOW_BIT|EGL_PIXMAP_BIT,
#else
		EGL_SURFACE_TYPE, EGL_WINDOW_BIT,
#endif
        EGL_NONE,
    };

    EGLConfig config;
    EGLint matchingConfigs;

    if (eglChooseConfig(_egl.eglDisplay,
                              configAttribs,
                              &config,
                              1,
                              &matchingConfigs))
    {
        if(matchingConfigs >= 1)
            return config;
    }

    return 0;
}

Visual *
createVisual(bool doubleBuffer, unsigned samples, Config *cnfg, Profile profile) {
    EGLint api_bits;
    DEBUG_ENTER();

    if(profile.api == glfeatures::API_GLES)
    {
	switch (profile.major) {
        case 1:
            api_bits = EGL_OPENGL_ES_BIT;
            break;
        case 3:
            if (has_EGL_KHR_create_context) {
                api_bits = EGL_OPENGL_ES3_BIT;
                break;
            }
            /* fall-through */
        case 2:
            api_bits = EGL_OPENGL_ES2_BIT;
            break;
        default:
            std::cerr << "error: profile api " << profile.api << " profile.major "<<profile.major<< "not supported\n";
	    exit(1);
        }
    }

    EGLConfig config;
    Visual *visual = NULL;

    if(cnfg)
        config = (EGLConfig)cnfg ->config;
    else
        config = findEglConfig();

    if(config != 0) {
#ifdef DEBUG_ENABLE		
        std::cerr << "eglChooseConfig ok\n";
#endif
        visual = new WlVisual(profile, config);
    }
    else {
        std::cerr << "error: failed to get egl config\n";
    }

    DEBUG_LEAVE();
    return visual;
}

Drawable *
createDrawable(const Visual *visual, int width, int height,
               const glws::pbuffer_info *pbInfo)
{
    DEBUG_ENTER();

    Drawable *d = new WlDrawable(visual, width, height, pbInfo);

    DEBUG_LEAVE();
    return d;
}

Context *
createContext(const Visual *_visual, Context *shareContext, bool debug)
{
    Profile profile = _visual->profile;
    const WlVisual *visual = static_cast<const WlVisual *>(_visual);
    EGLContext share_context = EGL_NO_CONTEXT;
    EGLContext context;
    Attributes<EGLint> attribs;

    DEBUG_ENTER();

    if (shareContext) {
        share_context = static_cast<WlContext*>(shareContext)->context;
    }

    int contextFlags = 0;
    if (profile.api == glfeatures::API_GL) {
        load("libGL.so.1");

        if (has_EGL_KHR_create_context) {
            attribs.add(EGL_CONTEXT_MAJOR_VERSION_KHR, profile.major);
            attribs.add(EGL_CONTEXT_MINOR_VERSION_KHR, profile.minor);
            int profileMask = profile.core ? EGL_CONTEXT_OPENGL_CORE_PROFILE_BIT_KHR : EGL_CONTEXT_OPENGL_COMPATIBILITY_PROFILE_BIT_KHR;
            attribs.add(EGL_CONTEXT_OPENGL_PROFILE_MASK_KHR, profileMask);
            if (profile.forwardCompatible) {
                contextFlags |= EGL_CONTEXT_OPENGL_FORWARD_COMPATIBLE_BIT_KHR;
            }
        } else if (profile.versionGreaterOrEqual(3, 2)) {
            std::cerr << "error: EGL_KHR_create_context not supported\n";
            return NULL;
        }
    } else if (profile.api == glfeatures::API_GLES) {
        if (profile.major >= 2) {
            load("libGLESv2.so.2");
        } else {
            load("libGLESv1_CM.so.1");
        }

        if (has_EGL_KHR_create_context) {
            attribs.add(EGL_CONTEXT_MAJOR_VERSION_KHR, profile.major);
            attribs.add(EGL_CONTEXT_MINOR_VERSION_KHR, profile.minor);
        } else {
            attribs.add(EGL_CONTEXT_CLIENT_VERSION, profile.major);
        }
    } else {
        assert(0);
        return NULL;
    }

    if (debug) {
        contextFlags |= EGL_CONTEXT_OPENGL_DEBUG_BIT_KHR;
    }
    if (contextFlags && has_EGL_KHR_create_context) {
        attribs.add(EGL_CONTEXT_FLAGS_KHR, contextFlags);
    }
    attribs.end(EGL_NONE);

    EGLenum api = translateAPI(profile);
    bindAPI(api);

    context = eglCreateContext(_egl.eglDisplay, visual->config, share_context, attribs);
    if (!context) {
        if (debug) {
            // XXX: Mesa has problems with EGL_CONTEXT_OPENGL_DEBUG_BIT_KHR
            // with OpenGL ES contexts, so retry without it
            return createContext(_visual, shareContext, false);
        }
        return NULL;
    }

    Context *ctxt = new WlContext(visual, context);

    DEBUG_LEAVE();

    return ctxt;
}

bool
makeCurrentInternal(Drawable *drawable, Context *context)
{
    DEBUG_ENTER();

    EGLBoolean ok;
	EGLSurface eglSurface;	

    if (!drawable || !context) {
#ifdef ENABLE_DEBUG		
        std::cerr << "makeCurrent: free\n";
#endif
        ok = eglMakeCurrent(_egl.eglDisplay, EGL_NO_SURFACE, EGL_NO_SURFACE, EGL_NO_CONTEXT);
    } else {
        WlDrawable *eglDrawable = static_cast<WlDrawable *>(drawable);
        WlContext *eglContext = static_cast<WlContext *>(context);

        eglSurface = eglDrawable->eglSurface;
        _egl.eglSurface = eglDrawable->eglSurface;

        EGLenum api = translateAPI(eglContext->profile);
        bindAPI(api);		

        if(use_wayland_backend) {
            eglSwapInterval(_egl.eglDisplay,0);
        }

		ok = eglMakeCurrent(_egl.eglDisplay, eglSurface,
								   eglSurface, eglContext->context); // FIXME!
        if (ok) {
            eglDrawable->api = api;
        }
    }

#ifdef DEBUG_ENABLE
    if(!ok)
        std::cerr << "makeCurrent failed\n";
    else
        std::cerr << "makeCurrent ok\n";	

    std::cerr << "Egl error: " << eglGetError() << "\n";
    std::cerr << "Gl error: " << glGetError() << "\n";
#endif	

    DEBUG_LEAVE();

    return ok;
}

bool
bindTexImage(Drawable *pBuffer, int iBuffer) {
    std::cerr << "error: EGL/XLIB::wglBindTexImageARB not implemented.\n";
    assert(pBuffer->pbuffer);
    return true;
}

bool
releaseTexImage(Drawable *pBuffer, int iBuffer) {
    std::cerr << "error: EGL/XLIB::wglReleaseTexImageARB not implemented.\n";
    assert(pBuffer->pbuffer);
    return true;
}

bool
setPbufferAttrib(Drawable *pBuffer, const int *attribList) {
    // nothing to do here.
    assert(pBuffer->pbuffer);
    return true;
}

bool
processEvents(void)
{
    return false;
}

}

